/////////////////////////////////////////////////////////////////////////////////
// KERNEL: Psychedelic-A.kernel //// Generic psychedelic shader "A" /////////////
/////////////////////////////////////////////////////////////////////////////////
// 
// This introduces multiple centers such that the pattern tiles the screen.
//
/////////////////////////////////////////////////////////////////////////////////

#define AUTOSCALE y=8.0
#define XFUNC(t,x,y) sin(x)+cos(y*4)

/////////////////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////
//
// The following shader kernel was intended to be used via an include directive
// in a shader fragment file of the form
//
//     #define parameter1 value
//     #define parameter2 value
//         etc.
//     #include TheEmu-A.fsh-kernel
//
// but #include is not supported by the shader language and so it has had to  be
// copied into each shader fragment file that uses it.  Rather than manualy copy
// it into each .fsh file, which would be a maintainance problem, I use a simple
// script file to concatenate it with each set of parameters to create  the  set
// of .fsh files.
//
/////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////
//
// This shader kernel operates using the following basic algorithm:-
//
//     Convert the pixel position to (x,y) co-ordinates with their
//     origin at the center of the screen with on screen positions
//     having x and y co-ordinates in the range -1.0 to +1.0.
//
//     Convert from (x,y) co-ordinates to polar co-ordinates (p,a)
//     where p is the distance from (x,y) = (0,0) and a the angle.
//     The radial co-ordinate is called p rather than the commonly
//     used r or R to avoid clashes with using r or R for red.
//
//     Apply a 'colourPick' function that returns a value, Q, that
//     is a time dependant function of (p,a).  This function is of
//     the form
//
//         Q(t,p,a) = a0 * ( cos ( x * a1 * cos(t/a2+a3) ) )
//                  + b0 * ( cos ( y * b1 * sin(t/b2+b3) ) )
//
//     where a0 to a3 and b0 to b3 are all constant  coefficients.
//     The function is applied up to eight times,  each  time with 
//     a different set of coefficients and the results are summed.
//
//     Assign Q, with optional weightings, to each of  QR, QG  and 
//     QB to give initial values for Red, Green and Blue values to
//     be 'mixed' by the next stage.
//
//     Apply a 'colorMix' function to determine the Red, Green and
//     Blue components of the output. This function is of the form
//
//         C(t,Q) = Q*c0 * ( c1 * cos(t/c2) + c3 * sin(t/c4) )
//
//     where c0, c1, c2, c3 and c4 are constant coefficients. The
//     function is applied three times for each of the R, G and B
//     components and the R, G and B results summed seperately. 
//
// However, various optional modifications can be applied to the basic algorithm
// outlined above. This enables such changing such things as
//
//        The speed 
//        The intensity of the output
//        The co-ordinate system's origin
//        The form of the co-ordinate transform
//
// all of which may be varied as functions of time and pixel position. The result
// is a very flexible shader fragment kernel that can  easily  be  configured  by 
// supplying a few tailoring parameters to overide their default values.
//
/////////////////////////////////////////////////////////////////////////////////
//
// Shader kernel tailoring parameters
//
// 1) Simple numerical shader parameters
//
//    These are all constants that are applied as multiplicative  factors  or
//    offsets within the shader. The defaults locate the center of the effect
//    produced by the shader at the center of the screen,  running at nominal
//    speed and with nomimal intensities for all colour components.
//
//        X0        : Center of X co-ordinate    - optional - defaults to 0.0
//        Y0        : Center of Y co-ordinate    - optional - defaults to 0.0
//
//        X1        : Scale for X co-ordinate    - optional - defaults to 1.0
//        Y1        : Scale for Y co-ordinate    - optional - defaults to 1.0
//
//        R0        : Basic colour factor, red   - optional - defaults to 1.0
//        G0        : Basic colour factor, green - optional - defaults to 1.0
//        B0        : Basic colour factor, blue  - optional - defaults to 1.0
//
//        SPEED     : Overall speed factor       - optional - defaults to 1.0
//        INTENSITY : Overall intensity factor   - optional - defaults to 1.0
//  
//    Additionally the AUTOSCALE parameter may be used to automatically scale
//    both the x and y coordinates such that, assuming square pixels, use the
//    same scale, i.e. the line x = y is at 45 degrees and x*x + y+y = k is a
//    circle on the screen. It takes the form
//
//        #define AUTOSCALE x = 1.0
//
//    or
//
//        #define AUTOSCALE y = 1.0
//
//    depending on which of the x or y co-ordinates is to to have its scale
//    fixed with the other co-ordinates scale being adjusted to match.
//
// 2) Sets of shader coefficients
//
//    These parameters comprise lists of numeric values passed as part of the
//    argument lists to the colourPick and colourMix functions.  The defaults
//    for these closely reproduce WyldAnimal's 2DShader-001 example.
// 
//        AA,BB,CC,DD,EE,..HH : Colour select coefficients - optional
//
//        mixRR, mixRG, mixRB : Colour mixing coefficients - optional
//        mixGR, mixGG, mixGB : Colour mixing coefficients - optional
//        mixBR, mixBG, mixBB : Colour mixing coefficients - optional
//
// 3) Co-ordinate transform functions
//
//    The colourPick function uses a calculation based on the pixel's position
//    expressed in polar co-ordinates. This is obtained using the cartesian to
//    polar co-ordinate transform defined by POLAR_P and POLAR_A.   By default
//    this is a simple (x,y) to polar co-ordinate transform, but may optionaly
//    perform a modified polar transform which may be be time dependant.
//
//        POLAR_P(t,x,y) : X,Y => Polar - optional - defaults to sqrt ( x*x + y*y )
//        POLAR_A(t,x,y) : X,Y => Polar - optional - defaults to atan ( x, y )
//
// 4) Modifier functions
//
//    Various aspects of the shader may be altered by defining modifier functions.
//    These modifiers allow variations to based on time and/or pixel position. The
//    defaults in each case are to do nothing.  Note that for the T, X, Y, P and A 
//    the modifier provides a new value for the corresponding variable but for the
//    I modifier it provides a multiplicative factor which is then applied to  the
//    previously calculated intensity.
//
//       TFUNC(t,x,y)   : Elapsed time modifier function         - optional
//
//       XFUNC(t,x,y)   : X,Y co-ordinates X modifier function   - optional
//       YFUNC(t,x,y)   : X,Y co-ordinates Y modifier function   - optional
//
//       PFUNC(t,p,a)   : Polar co-ordinates P modifier function - optional
//       AFUNC(t,p,a)   : Polar co-ordinates A modifier function - optional
//
//       IFUNCXY(t,x,y) : Overall intensity modifier function    - optional
//       IFUNCPA(t,p,a) : Overall intensity modifier function    - optional
//
// 5) RGB Components
//
//   The calculations eventualy result in a set of Red, Blue and Green colour
//   components, R, G and B plus an intensity. By default R, G and B are used
//   unmodified for the red, blue and green components of the output from the
//   shader, but this can be modified by defining RGB_COMPONENTS. The default
//   for this is R, G, B, but the order may be changed and each component may
//   be an arbitrary, possibly time dependant, function of R, G and B.
//   
//
/////////////////////////////////////////////////////////////////////////////////

// Defaults for the simple numerical shader parameters

#ifndef X0
#define X0 0.0
#endif

#ifndef Y0
#define Y0 0.0
#endif

#ifndef X1
#define X1 1.0
#endif

#ifndef Y1
#define Y1 1.0
#endif

#define R0 1.0
#define G0 1.0
#define B0 1.0

#ifndef SPEED
#define SPEED 1.0
#endif 
 
#ifndef INTENSITY
#define INTENSITY 1.0
#endif
 
#ifndef RGB_COMPONENTS
#define RGB_COMPONENTS R, G, B
#endif

/////////////////////////////////////////////////////////////////////////////////

// Defaults for the colour selection coefficients. These defaults are only used
// if no colour selection coeffients have been supplied. For the convenience of
// the users the default sets of coefficients are visible to them and so may be
// used when defining their own sets of coefficients.

#define DEFAULT_AA  1.0, 80.0, 15.0, 0.0,  1.0, 40.0, 10.0, 0.0
#define DEFAULT_BB  1.0, 10.0, 15.0, 0.0,  1.0, 40.0, 25.0, 0.0
#define DEFAULT_CC  1.0, 10.0,  5.0, 0.0,  1.0, 80.0, 35.0, 0.0

#ifndef AA
#ifndef BB
#ifndef CC
#ifndef DD

#ifndef EE
#ifndef FF
#ifndef GG
#ifndef HH

   #define AA DEFAULT_AA
   #define BB DEFAULT_BB
   #define CC DEFAULT_CC

#endif
#endif
#endif
#endif

#endif
#endif
#endif
#endif

// Defaults for the colour mixing coefficients. These are only used if
// no colour mixing coefficients have been supplied.  Note  that it is
// not necessary to define all nine sets of colour mixing coefficients
// as any set that is not defined is not used,  effectively having the
// value 0.0, 0.0, 1.0, 0.0, 1.0 but optimised away.

#ifndef mixRR
#ifndef mixRG
#ifndef mixRB

#ifndef mixGR
#ifndef mixGG
#ifndef mixGB

#ifndef mixBR
#ifndef mixBG
#ifndef mixBB

   #define mixRR  0.5,    1.0, 10.0,   0.0, 1.0
   #define mixGR  0.25,   1.0, 10.0,   0.0, 1.0
   #define mixBB  0.75,   0.0,  1.0,   1.0, 1.0

#endif
#endif
#endif

#endif
#endif
#endif

#endif
#endif
#endif

/////////////////////////////////////////////////////////////////////////////////

// Defaults for the co-ordinate transform functions.

#ifndef POLAR_P
#define POLAR_P(t,x,y) sqrt ( x*x + y*y )
#endif

#ifndef POLAR_A
#define POLAR_A(t,x,y) atan ( x, y )
#endif

/////////////////////////////////////////////////////////////////////////////////

// The current set of modifier functions default to no-action and require no
// explicit default definitions. They are simply not applied if they are not
// defined.

/////////////////////////////////////////////////////////////////////////////////

// Explicit inputs

uniform float u_Elapsed;   // The elapsed time in seconds
uniform vec2 u_WindowSize; // Window dimensions in pixels

// Implicit inputs
//    gl_FragCoord
//
// Outputs 
//    gl_FragColor

/////////////////////////////////////////////////////////////////////////////////

float colourPick
 ( float t,
   float p,
   float a,
   float a0, float a1, float a2, float a3,
   float b0, float b1, float b2, float b3
 )
 { float q = 0.0;
   if ( a0 != 0.0 ) q += a0 * ( cos ( p * a1 * cos(t/a2+a3) ) );
   if ( a0 != 0.0 ) q += b0 * ( cos ( a * b1 * sin(t/b2+b3) ) );
   return q;
 }

float colourMix 
 ( float t,
   float Q,
   float c0, float c1, float c2, float c3, float c4
 )
 { float q = Q*c0;
   if ( q != 0.0 ) q *= ( c1 * cos(t/c2) + c3 * sin(t/c4) );
   return q;
 }

/////////////////////////////////////////////////////////////////////////////////

void main( void )
 {

   // The time, with optional speed factor.

   float t = u_Elapsed * SPEED;

   // Rescale and re-center the pixel's co-ordinates.

   vec2 position = ( gl_FragCoord.xy / u_WindowSize.xy ); // Range 0 to 1

   #if defined AUTOSCALE

      float xy_ratio = u_WindowSize.xy.x / u_WindowSize.xy.y;

      vec2 autoscale = vec2 ( 0.0, 0.0 );
      autoscale.AUTOSCALE;

      if ( autoscale.x==0.0 ) autoscale.x = autoscale.y * xy_ratio;
      if ( autoscale.y==0.0 ) autoscale.y = autoscale.x / xy_ratio;
 
      float x0 = ( position.x * 2.0 - 1.0 ) * X1 * autoscale.x - X0;
      float y0 = ( position.y * 2.0 - 1.0 ) * Y1 * autoscale.y - Y0;

   #else

      float x0 = ( position.x * 2.0 - 1.0 ) * X1 - X0; // Range -X1 to +X1
      float y0 = ( position.y * 2.0 - 1.0 ) * Y1 - Y0; // Range -Y1 to +Y1

   #endif

   float x = x0;
   float y = y0; 

   // Apply any modifier functions for t, x and y.

   #if defined TFUNC
   t = TFUNC(t,x0,y0);
   #endif

   #if defined XFUNC
   x = XFUNC(t,x0,y0);
   #endif

   #if defined YFUNC
   y = YFUNC(t,x0,y0);
   #endif

   // Convert to polar co-ordinates

   float p0 = POLAR_P ( t, x, y );
   float a0 = POLAR_A ( t, x, y );

   float p = p0;
   float a = a0;

   #if defined PFUNC
   p = PFUNC(t,p0,a0);
   #endif

   #if defined AFUNC
   a = AFUNC(t,p0,a0);
   #endif

   // Calculate the colour of the pixel

   float Q = 0.0;

   #if defined AA
   Q += colourPick ( t, p, a, AA );
   #endif

   #if defined BB
   Q += colourPick ( t, p, a, BB );
   #endif

   #if defined CC
   Q += colourPick ( t, p, a, CC );
   #endif

   #if defined DD
   Q += colourPick ( t, p, a, DD );
   #endif

   #if defined EE
   Q += colourPick ( t, p, a, EE );
   #endif

   #if defined FF
   Q += colourPick ( t, p, a, FF );
   #endif

   #if defined GG
   Q += colourPick ( t, p, a, GG );
   #endif

   #if defined HH
   Q += colourPick ( t, p, a, HH );
   #endif

   // Determine the R, G and B components

   float QR = Q * R0;
   float QG = Q * G0;
   float QB = Q * B0;

   float R = 0.0;
   float G = 0.0;
   float B = 0.0;

   #if defined mixRR
   R += colourMix ( t, QR, mixRR );
   #endif

   #if defined mixRG
   R += colourMix ( t, QG, mixRG );
   #endif

   #if defined mixRB
   R += colourMix ( t, QB, mixRB );
   #endif

   #if defined mixGR
   G += colourMix ( t, QR, mixGR );
   #endif

   #if defined mixGG
   G += colourMix ( t, QG, mixGG );
   #endif

   #if defined mixGB
   G += colourMix ( t, QB, mixGB );
   #endif

   #if defined mixBR
   B += colourMix ( t, QR, mixBR );
   #endif

   #if defined mixBG
   B += colourMix ( t, QG, mixBG );
   #endif

   #if defined mixBB
   B += colourMix ( t, QB, mixBB );
   #endif

   // Update the output

   float intensity = INTENSITY;

   #if defined IFUNCXY
   intensity *= IFUNCXY(t,x,y);
   #endif

   #if defined IFUNCPA
   intensity *= IFUNCPA(t,p,a);
   #endif

   gl_FragColor = vec4 ( vec3 ( RGB_COMPONENTS ), intensity );

}

/////////////////////////////////////////////////////////////////////////////////
